/*
  Made by KinCony IoT: https://www.kincony.com

  Program functionality:
  This program uses ESP32-S3 to control two PCA9555 I/O expander chips via I2C, 
  controlling a total of 32 relays. The first PCA9555 chip has an I2C address of 0x21 
  and controls relays 1-16, while the second PCA9555 chip has an I2C address of 0x22 
  and controls relays 17-32.

  The relays are turned ON (set to low) and OFF (set to high) in sequence with a small 
  delay between each operation. The program reads and prints the current state of the 
  relays to the serial monitor after each change.
*/

#include <PCA95x5.h>

// Initialize two PCA9555 objects for controlling 32 relays
PCA9555 ioex1; // 1-16 relays (I2C address 0x21)
PCA9555 ioex2; // 17-32 relays (I2C address 0x22)

void setup() {
    // Start serial communication for debugging
    Serial.begin(115200);
    delay(10);

    // Initialize the I2C bus with GPIO 11 as SDA and GPIO 10 as SCL, 100kHz frequency
    Wire.begin(11, 10, 100000);

    // Configure the first PCA9555 chip (1-16 relays)
    ioex1.attach(Wire, 0x21);  // Set I2C address to 0x21
    ioex1.polarity(PCA95x5::Polarity::ORIGINAL_ALL);  // Set polarity to normal (no inversion)
    ioex1.direction(PCA95x5::Direction::OUT_ALL);  // Set all pins as output
    ioex1.write(PCA95x5::Level::H_ALL);  // Initialize all outputs to HIGH (relays OFF)

    // Configure the second PCA9555 chip (17-32 relays)
    ioex2.attach(Wire, 0x22);  // Set I2C address to 0x22
    ioex2.polarity(PCA95x5::Polarity::ORIGINAL_ALL);  // Set polarity to normal (no inversion)
    ioex2.direction(PCA95x5::Direction::OUT_ALL);  // Set all pins as output
    ioex2.write(PCA95x5::Level::H_ALL);  // Initialize all outputs to HIGH (relays OFF)

    delay(50);  // Wait for settings to take effect
}

void loop() {
    // Control the first PCA9555 chip (1-16 relays)
    for (size_t i = 0; i < 16; ++i) {
        Serial.print("set port low (chip 1): ");  // Turning ON the relay
        Serial.println(i);

        ioex1.write((PCA95x5::Port::Port)i, PCA95x5::Level::L);  // Set the relay to LOW (ON)
        Serial.println(ioex1.read(), BIN);  // Print the current state of all relays (in binary)
        delay(50);  // Small delay between each relay activation
    }

    // Control the second PCA9555 chip (17-32 relays)
    for (size_t i = 0; i < 16; ++i) {
        Serial.print("set port low (chip 2): ");  // Turning ON the relay
        Serial.println(i + 16);  // Display relay numbers 17-32

        ioex2.write((PCA95x5::Port::Port)i, PCA95x5::Level::L);  // Set the relay to LOW (ON)
        Serial.println(ioex2.read(), BIN);  // Print the current state of all relays (in binary)
        delay(50);  // Small delay between each relay activation
    }

    // Turn off the first PCA9555 chip (1-16 relays)
    for (size_t i = 0; i < 16; ++i) {
        Serial.print("set port high (chip 1): ");  // Turning OFF the relay
        Serial.println(i);

        ioex1.write((PCA95x5::Port::Port)i, PCA95x5::Level::H);  // Set the relay to HIGH (OFF)
        Serial.println(ioex1.read(), BIN);  // Print the current state of all relays (in binary)
        delay(50);  // Small delay between each relay deactivation
    }

    // Turn off the second PCA9555 chip (17-32 relays)
    for (size_t i = 0; i < 16; ++i) {
        Serial.print("set port high (chip 2): ");  // Turning OFF the relay
        Serial.println(i + 16);  // Display relay numbers 17-32

        ioex2.write((PCA95x5::Port::Port)i, PCA95x5::Level::H);  // Set the relay to HIGH (OFF)
        Serial.println(ioex2.read(), BIN);  // Print the current state of all relays (in binary)
        delay(50);  // Small delay between each relay deactivation
    }
}
